/* eslint-disable */
const tarball = require('tarball-extract');
const fs = require('fs');
const tmp = require('tmp');
const ref = require('ref');
const ffi = require('ffi');
const StructType = require('ref-struct');
const ArrayType = require('ref-array');
const path = require('path');
const util = require('util');
const electron = require('electron');
const settings = require('electron-settings');
const fsPromises = require('fs').promises;
const { StringDecoder } = require('string_decoder');
const pskey_struct_definitions = require('./pskey_struct_definitions');

/* interval between successive usb device polling, in ms */
const USB_POLLING_INTERVAL_MS = 500;
/* minimum time, in seconds, to display the "device connecting" window */
/* need to wait this much time before reporting this device is fully connected */
const DEVICE_CONNECTING_MIN_SEC = 3;
const DEVICE_CONNECTING_MIN_INTERVALS =
    (DEVICE_CONNECTING_MIN_SEC * 1000) / USB_POLLING_INTERVAL_MS;
/* maximum time, in seconds, to display the "device connecting" window */
/* need to wait this much time before removing the "device connected" window back to the "unplugged" window */
const DEVICE_CONNECTING_MAX_SEC = 10;
const DEVICE_CONNECTING_MAX_INTERVALS =
    (DEVICE_CONNECTING_MAX_SEC * 1000) / USB_POLLING_INTERVAL_MS;

const current_app = 'updater';
const {
    HEADSET_STATUS,
    MODEL_DEVICE_ERROR,
    MODEL_DEVICE_NOT_SUPPORTED,
    MODEL_DEVICE_UNKNOWN,
    MODELS
} = require('./consts');
const logger = require('../Logger').Logger;

const CARDO_VID = 0x2685;
const CARDO_MOTO_PIDS = [0x900, 0x901, 0x1200, 0x1201];
const CARDO_QBTM_PID = 0x1500;

const cfuContext = ref.types.void;
const cfuContextPtr = ref.refType(cfuContext);
const uint8Ptr = ref.refType('uint8');
const uint16Ptr = ref.refType('uint16');

const libusbDevice = ref.types.void;
const libusbDevicePtr = ref.refType(libusbDevice);
const libusbDevicePtrPtr = ref.refType(libusbDevicePtr);
const libusbDevicePtrPtrPtr = ref.refType(libusbDevicePtrPtr);
const libusbDevicePtrArray = ArrayType(libusbDevicePtr);
const libusbDeviceDescriptor = StructType({
    bLength: 'uint8',
    bDescriptorType: 'uint8',
    bcdUSB: 'uint16',
    bDeviceClass: 'uint8',
    bDeviceSubClass: 'uint8',
    bDeviceProtocol: 'uint8',
    bMaxPacketSize0: 'uint8',
    idVendor: 'uint16',
    idProduct: 'uint16',
    bcdDevice: 'uint16',
    iManufacturer: 'uint8',
    iProduct: 'uint8',
    iSerialNumber: 'uint8',
    bNumConfigurations: 'uint8'
});
const libusbDeviceDescriptorPtr = ref.refType(libusbDeviceDescriptor);

const cfuBasePath =
    process.env.NODE_ENV === 'development'
        ? './prebuilt/cfu/'
        : './library/headset/prebuilt/cfu/';
const libcfuDir = path.join(__dirname, cfuBasePath, process.platform);
const libcfuPath = path.join(libcfuDir, 'libcfu');
const libusbPath = path.join(libcfuDir, 'libusb-1.0');

if (process.platform === 'win32') {
    var setdl = ffi.Library('kernel32', {
        SetDllDirectoryA: ['bool', ['string']]
    });
    setdl.SetDllDirectoryA(libcfuDir);
}

let FLASHER_FUNC = null;

var libcfu = ffi.Library(libcfuPath, {
    cfu_init: ['int32', ['string', 'uint32']],
    cfu_open: [cfuContextPtr, ['pointer']],
    cfu_reset: ['int32', [cfuContextPtr]],
    cfu_close: ['void', [cfuContextPtr]],
    cfu_status: ['int32', [cfuContextPtr]],
    cfu_read_pskey: ['int32', [cfuContextPtr, 'uint16', uint16Ptr, 'uint32']],
    cfu_write_pskey: ['int32', [cfuContextPtr, 'uint16', uint16Ptr, 'uint32']],
    cfu_clear_pskey: ['int32', [cfuContextPtr, 'uint16']],
    cfu_flash_btm: ['int32', [cfuContextPtr, uint8Ptr, 'uint32', 'pointer']],
    cfu_flash_dsp: ['int32', [cfuContextPtr, uint8Ptr, 'uint32', 'pointer']],
    cfu_flash_sqif: ['int32', [cfuContextPtr, uint8Ptr, 'uint32', 'pointer']],
    cfu_write_dsp: [
        'int32',
        [cfuContextPtr, 'uint32', uint8Ptr, 'uint32', 'pointer']
    ],
    cfu_get_config: ['int32', [cfuContextPtr, 'uint16', uint16Ptr, 'uint32']],
    cfu_flash_qbtm: ['int32', [cfuContextPtr, uint8Ptr, 'uint32', 'pointer']]
});

var logPath = path.join(
    electron.app ? electron.app.getPath('userData') : app.getPath('userData'),
    'cfu.log'
);
var ret = libcfu.cfu_init(logPath, 5);
console.log('cfu_init: ' + ret);

var libusb = ffi.Library(libusbPath, {
    libusb_init: ['int32', ['int32']],
    libusb_get_device_list: ['int32', ['int32', libusbDevicePtrPtrPtr]],
    libusb_free_device_list: ['void', [libusbDevicePtrPtr, 'int32']],
    libusb_get_device_descriptor: [
        'int32',
        [libusbDevicePtr, libusbDeviceDescriptorPtr]
    ]
});

ret = libusb.libusb_init(0);
console.log('libusb_init: ' + ret);

function _flash_loader(cfu, data, len, cfu_progress_cb, cfu_complete_cb) {
    // The TI dsp loader is written to flash address 0x0h using the cfu writedsp command
    libcfu.cfu_write_dsp.async(
        cfu,
        0,
        data,
        len,
        cfu_progress_cb,
        cfu_complete_cb
    );
}

FLASHER_FUNC = {
    'app.dfu': libcfu.cfu_flash_btm.async,
    'dsp.img': libcfu.cfu_flash_dsp.async,
    'ldr.bin': _flash_loader,
    'ext.pck': libcfu.cfu_flash_sqif.async,
    'ver.bin': libcfu.cfu_flash_qbtm.async
};

const QCC_LANGUAGE = {
    0: 'en_us',
    1: 'en_uk',
    2: 'sp',
    3: 'fr',
    4: 'de',
    5: 'it',
    6: 'pt',
    7: 'nl',
    8: 'ru',
    9: 'ja',
    10: 'ch',
    11: 'he',
    12: 'ko'
};

// Configuration blocks for QCC devices:
// https://cardovsts.visualstudio.com/Cardo/_wiki/wikis/Cardo/391/Configuration-Blocks-as-Interfaces-for-QCC
const CONF_BLOCK_DEV_UID = 57;
const CONF_BLOCK_VERSIONS = 33;
const CONF_BLOCK_BT_NAME = 43;
const CONF_BLOCK_RESERVED = 69;

const PSKEYS_BASE = 650;
const PSKEY_BTM_VERSION = PSKEYS_BASE + 16;
const PSKEY_DEV_UID = PSKEYS_BASE + 18;
const PSKEY_PUB_FW_VER = PSKEYS_BASE + 37;
const PSKEY_BT_NAME = PSKEYS_BASE + 38;
const PSKEY_TO_RESET = PSKEYS_BASE + 42;

// Size in words for each PSKEY we're interested in
// Note: PSKEY_PUB_FW_VER depends on the device type and will be updated in run-time
var PSKEY_SIZE = {
    PSKEY_DEV_UID: 8,
    PSKEY_PUB_FW_VER: 0,
    PSKEY_BT_NAME: 32,
    PSKEY_BTM_VERSION: 8,
    PSKEY_TO_RESET: 16,
    QCC_PUB_FW_VER: 6,
    QCC_BT_NAME: 20,
    QCC_RESERVED: 2
};

const CARDO_DEV_INFO = {
    0x5052: { name: MODELS.PRO1, fwver_size: 2 },
    0x424a: { name: MODELS.PACKTALK_BOLD_JBL, fwver_size: 4 },
    0x534a: { name: MODELS.PACKTALK_SLIM_JBL, fwver_size: 4 },
    0x5042: { name: MODELS.PACKTALK_BOLD, fwver_size: 4 },
    0x5053: { name: MODELS.PACKTALK_SLIM, fwver_size: 4 },
    0x5047: { name: MODELS.PACKTALK_SKI, fwver_size: 4 },
    0x4c31: { name: MODELS.PACKTALK, fwver_size: 4 },
    0x4843: { name: MODELS.SMART_H, fwver_size: 4 },
    0x504c: { name: MODELS.SMARTPACK, fwver_size: 4 },
    0x4657: { name: MODELS.FREECOM4PLUS, fwver_size: 4 },
    0x4652: { name: MODELS.FREECOM4PLUSCR, fwver_size: 4 },
    0x4634: { name: MODELS.FREECOM4, fwver_size: 4 },
    0x4645: { name: MODELS.FREECOM2PLUS, fwver_size: 4 },
    0x4632: { name: MODELS.FREECOM2, fwver_size: 4 },
    0x4650: { name: MODELS.FREECOM1PLUS, fwver_size: 4 },
    0x464d: { name: MODELS.FREECOM1FM, fwver_size: 4 },
    0x4631: { name: MODELS.FREECOM1, fwver_size: 4 },
    0x5452: { name: MODELS.TERRANOXT, fwver_size: 4 },
    0x444a: { name: MODELS.PACKTALK_DUCATI, fwver_size: 4 },
    0x4653: { name: MODELS.FREECOM_SOLO, fwver_size: 4 },
    0x504b: { name: MODELS.PACKTALK_BLACK, fwver_size: 4 },
    0x4147: { name: MODELS.PACKTALK_SLIM_AGV, fwver_size: 4 },
    0x4d43: { name: MODELS.MCLAREN_INTERCOM, fwver_size: 4 },
    0x424c: { name: MODELS.PACKTALK_LOUIS, fwver_size: 4 },
    0x4246: { name: MODELS.PACKTALK_BOLD_JBL_ICS00177, fwver_size: 4 },
    0x504d: { name: MODELS.SKI_II_SINGLE_PACK, fwver_size: 4 },
    0x504e: { name: MODELS.PACKTALK_BLACK_ICS00177, fwver_size: 4 },
    0x534b: { name: MODELS.PACKTALK_SLIM_JBL_ICS00177, fwver_size: 4 },
    0x4d4c: { name: MODELS.PACKTALK_SLIM_MCLAREN_ICS00177, fwver_size: 4 },
    0x424d: { name: MODELS.PACKTALK_BOLD_LOUIS_NEW_FM_ICS, fwver_size: 4 },
    0x534c: { name: MODELS.SPIRIT, fwver_size: 4 },
    0x5353: { name: MODELS.SPIRIT_HD, fwver_size: 4 },
    0x5854: { name: MODELS.FRC2X, fwver_size: 4 },
    0x5846: { name: MODELS.FRC4X, fwver_size: 4 },
    0x5054: { name: MODELS.PACKTALK_EDGE, fwver_size: 4 },
    0x5044: { name: MODELS.PACKTALK_EDGE_DUCATI, fwver_size: 4 },
    0x4b50: { name: MODELS.PACKTALK_EDGE_KTM, fwver_size: 4 },
    0x5048: { name: MODELS.PACKTALK_EDGE_HONDA, fwver_size: 4 },
    0x4f43: { name: MODELS.PT_OUTDOOR_CONSUMER, fwver_size: 4 },
    0x4f54: { name: MODELS.PT_OUTDOOR_INSTRUCTOR, fwver_size: 4 },
    0x4f53: { name: MODELS.PT_OUTDOOR_STUDENT, fwver_size: 4 },
    0x5043: { name: MODELS.PT_CUSTOM, fwver_size: 4 },
    0x5349: { name: MODELS.PT_EDGE_SIMPSON, fwver_size: 4 },
    0x504f: { name: MODELS.PACKTALK_NEO, fwver_size: 4 },
    0x5450: { name: MODELS.PACKTALK_EDGE, fwver_size: 4 },
    0x5444: { name: MODELS.PACKTALK_EDGE_DUCATI, fwver_size: 4 },
    0x544b: { name: MODELS.PACKTALK_EDGE_KTM, fwver_size: 4 },
    0x544e: { name: MODELS.PACKTALK_EDGE_HONDA, fwver_size: 4 },
    0x5453: { name: MODELS.PT_EDGE_SIMPSON, fwver_size: 4 },
    0x544f: { name: MODELS.PACKTALK_ORV, fwver_size: 4 },
    0x4f52: { name: MODELS.PACKTALK_ORV, fwver_size: 4 },
    0x4456: { name: MODELS.PACKTALK_DELTA_V, fwver_size: 4 },
    0x5543: { name: MODELS.UCS_LS2, fwver_size: 4 },
    0x4548: { name: MODELS.PACKTALK_EDGE_HARLEY, fwver_size: 4 },
    0x3248: { name: MODELS.FREECOM2X_HARLEY, fwver_size: 4 },
    0x3448: { name: MODELS.FREECOM4X_HARLEY, fwver_size: 4 },
    0x4e45: { name: MODELS.PACKTALK_NEO_SE, fwver_size: 4 },
    0x4553: { name: MODELS.PACKTALK_EDGE_NO_FPGA, fwver_size: 4 },
    0x5050: { name: MODELS.PACKTALK_PRO, fwver_size: 4 },
    0x464d: { name: MODELS.FC_MOTO_UCS, fwver_size: 4 },
    0x4756: { name: MODELS.GIVI_UCS, fwver_size: 4 },
    0x5252: { name: MODELS.SHOCKWAVE_MESH, fwver_size: 4 },
    0x3542: { name: MODELS.M509_UCS_SPIRIT, fwver_size: 4 },
    0x4e46: { name: MODELS.PACKTALK_NEO_NO_FPGA, fwver_size: 4 },
    0x504c: { name: MODELS.PACKTALK_NEO_LOUIS, fwver_size: 4 },
    0x4f46: { name: MODELS.PACKTALK_OUTDOOR_FREEJUMP, fwver_size: 4 },
    0x5350: { name: MODELS.SCHUBERTH_SC_EDGE, fwver_size: 4 },
    0x5049: { name: MODELS.PACKTALK_NEO_LOUIS, fwver_size: 4 }
};

const DEV_MODE = {
    0: 'unplugged',
    1: 'btm',
    2: 'dfu',
    3: 'sqif',
    4: 'dsp',
    5: 'qbtmnormal',
    6: 'qbtmconfig'
};

function parse_pskey_dev_uid(headset, buf) {
    //first word is the Unique Device ID
    var dev_id = headset.is_qcc ? buf.readInt16BE(0) : buf.readInt16LE(0);
    logger.info('parse_pskey_dev_uid - dev_id = ', dev_id);
    if (!(dev_id in CARDO_DEV_INFO))
        return [MODEL_DEVICE_NOT_SUPPORTED, 'XXXXXXXX'];

    var dev_name = CARDO_DEV_INFO[dev_id].name;

    // size of the following pskey depends on the type of the device we have
    // set it now globally, so later users of it won't have to care
    PSKEY_SIZE.PSKEY_PUB_FW_VER = CARDO_DEV_INFO[dev_id].fwver_size;

    // Rest of the buffer is a string containing the serial number,
    // but we need to swap the bytes first before extracting string
    buf.swap16();
    const decoder = new StringDecoder('ascii');

    // Calculate index of the last serial digit (i.e. ignore zeroes)
    var last_digit = buf.indexOf(0);
    if (last_digit == -1) last_digit = buf.length;

    // Trim tailing zeros
    buf = buf.slice(0, last_digit);

    var dev_serial = decoder.write(buf);

    if (headset.is_qcc) {
        // For QCC devices - saved as BE, not LE
        let x = '';
        for (var i = 0; i < dev_serial.length; i += 2) {
            x += dev_serial[i + 1] + dev_serial[i];
        }
        dev_serial = x;
    }

    logger.info(
        'parse_pskey_dev_uid: prev dev_id =',
        headset.dev_id,
        'new dev_id =',
        dev_id
    );
    logger.info(
        'parse_pskey_dev_uid: prev dev_serial =',
        headset.dev_serial,
        'new dev_serial =',
        dev_serial
    );
    // Remember our dev_id and serial number, will be used on flashing
    headset.dev_id = dev_id;
    headset.dev_serial = dev_serial;

    return [dev_name, dev_serial];
}

function retry_cfu_open(headset) {
    headset.timer = setTimeout(() => {
        libcfu.cfu_open.async(null, function(err, cfu) {
            cfu_open_cb(headset, err, cfu);
        });
    }, USB_POLLING_INTERVAL_MS);
}

function retry_cfu_status(headset) {
    headset.timer = setTimeout(() => {
        if (!headset.cfu_context) {
            // Device is disconnected
            return;
        }

        if (headset.is_qcc) {
            var timer = setTimeout(() => {
                const isConnected = is_device_connected(headset);
                console.log('retry_cfu_status: is_connected = ', isConnected);

                if (!isConnected) {
                    // Device is disconnected
                    maybe_report_old_headset(headset);
                    return;
                } else {
                    retry_cfu_status(headset);
                }
            }, 1000);
        }

        const lastCallingCfuStatus = Math.floor(Math.random() * 100000);
        headset.calling_cfu_status = lastCallingCfuStatus;
        libcfu.cfu_status.async(headset.cfu_context, function(err, status) {
            if (headset.calling_cfu_status !== lastCallingCfuStatus) return;
            if (headset.is_qcc) clearTimeout(timer);
            cfu_status_cb(headset, err, status);
        });
    }, USB_POLLING_INTERVAL_MS);
}

function is_device_connected(headset) {
    var listPtrPtrPtr = ref.alloc(libusbDevicePtrPtr, null);
    var num_devices = libusb.libusb_get_device_list(0, listPtrPtrPtr);
    logger.info(
        util.format(
            'libusb_get_device_list: %d',
            num_devices,
            headset.is_qcc,
            headset.switching_mode,
            headset.flashing_fw
        )
    );

    if (headset.is_qcc && (headset.switching_mode || headset.flashing_fw)) {
        logger.info(
            'is_device_connected: QCC device switching modes - not checking connection',
            headset.switching_mode,
            headset.flashing_fw
        );
        return true;
    }

    var listPtrPtr = listPtrPtrPtr.deref();

    var usb_devices = new libusbDevicePtrArray(
        listPtrPtr.reinterpret(ref.sizeof.pointer * num_devices)
    );
    var usb_descriptor = new libusbDeviceDescriptor();

    for (var i = 0; i < usb_devices.length; i++) {
        var devicePtr = usb_devices[i];

        // reset struct values to 0
        usb_descriptor.ref().fill(0);

        var ret = libusb.libusb_get_device_descriptor(
            devicePtr,
            usb_descriptor.ref()
        );
        if (ret) {
            logger.info(
                util.format('libusb_get_device_descriptor failed: %d', ret)
            );
            continue;
        }

        var vidHex = ('0000' + usb_descriptor.idVendor.toString(16)).substr(-4);
        var pidHex = ('0000' + usb_descriptor.idProduct.toString(16)).substr(
            -4
        );

        if (usb_descriptor.idVendor == CARDO_VID) {
            // Found a Cardo device
            return true;
        }
    }

    return false;
}

function is_qcc_device() {
    var listPtrPtrPtr = ref.alloc(libusbDevicePtrPtr, null);
    var num_devices = libusb.libusb_get_device_list(0, listPtrPtrPtr);
    logger.info(util.format('libusb_get_device_list: %d', num_devices));

    var listPtrPtr = listPtrPtrPtr.deref();

    var usb_devices = new libusbDevicePtrArray(
        listPtrPtr.reinterpret(ref.sizeof.pointer * num_devices)
    );
    var usb_descriptor = new libusbDeviceDescriptor();

    for (var i = 0; i < usb_devices.length; i++) {
        var devicePtr = usb_devices[i];

        // reset struct values to 0
        usb_descriptor.ref().fill(0);

        var ret = libusb.libusb_get_device_descriptor(
            devicePtr,
            usb_descriptor.ref()
        );
        if (ret) {
            logger.info(
                util.format('libusb_get_device_descriptor failed: %d', ret)
            );
            continue;
        }

        var vidHex = ('0000' + usb_descriptor.idVendor.toString(16)).substr(-4);
        var pidHex = ('0000' + usb_descriptor.idProduct.toString(16)).substr(
            -4
        );

        logger.info(
            util.format(
                'Found USB device Vendor:Device = %s:%s',
                vidHex,
                pidHex
            )
        );

        if (usb_descriptor.idVendor == CARDO_VID) {
            if (usb_descriptor.idProduct == CARDO_QBTM_PID) {
                logger.info('Found QCC device');
                return true;
            }
        }
    }

    return false;
}

function find_unsupported_units() {
    var found_unsupported_cardo = false;
    var listPtrPtrPtr = ref.alloc(libusbDevicePtrPtr, null);
    var num_devices = libusb.libusb_get_device_list(0, listPtrPtrPtr);
    logger.info(util.format('libusb_get_device_list: %d', num_devices));

    var listPtrPtr = listPtrPtrPtr.deref();

    var usb_devices = new libusbDevicePtrArray(
        listPtrPtr.reinterpret(ref.sizeof.pointer * num_devices)
    );
    var usb_descriptor = new libusbDeviceDescriptor();

    for (var i = 0; i < usb_devices.length; i++) {
        var devicePtr = usb_devices[i];

        // reset struct values to 0
        usb_descriptor.ref().fill(0);

        var ret = libusb.libusb_get_device_descriptor(
            devicePtr,
            usb_descriptor.ref()
        );
        if (ret) {
            logger.info(
                util.format('libusb_get_device_descriptor failed: %d', ret)
            );
            continue;
        }

        var vidHex = ('0000' + usb_descriptor.idVendor.toString(16)).substr(-4);
        var pidHex = ('0000' + usb_descriptor.idProduct.toString(16)).substr(
            -4
        );

        logger.info(
            util.format(
                'Found USB device Vendor:Device = %s:%s',
                vidHex,
                pidHex
            )
        );

        if (usb_descriptor.idVendor == CARDO_VID) {
            // we may have potentially found an unsupported cardo device
            found_unsupported_cardo = true;
            // check if this cardo device is supported - if it does - clear the found_unsupported_cardo flag
            CARDO_MOTO_PIDS.forEach(function(pid) {
                if (usb_descriptor.idProduct == pid) {
                    found_unsupported_cardo = false;
                }
            });
            if (usb_descriptor.idProduct == CARDO_QBTM_PID) {
                logger.info('Found QCC device');
                found_unsupported_cardo = false;
            }

            logger.info(
                util.format(
                    'found cardo device! old = %d, pid = %s',
                    found_unsupported_cardo,
                    pidHex
                )
            );
        }
    }

    logger.info(
        util.format('find_unsupported_units: %d', found_unsupported_cardo)
    );
    return found_unsupported_cardo;
}

function maybe_report_old_headset(headset) {
    let connect_status;

    if (find_unsupported_units(headset)) {
        logger.info('Found an unsupported cardo device');
        headset.status = HEADSET_STATUS.HEADSET_UNSUPPORTED;
        connect_status = HEADSET_STATUS.HEADSET_CONNECTED;
    } else {
        logger.info("Didn't find any unsupported cardo device");
        connect_status = headset.status = HEADSET_STATUS.HEADSET_DISCONNECTED;
    }

    headset.connectionCallback(
        connect_status,
        false /* dfu */,
        headset.status == HEADSET_STATUS.HEADSET_UNSUPPORTED
    );
}

function handle_unplugged_headset(headset) {
    logger.info('handle_unplugged_headset', headset, headset.cfu_context);
    let old_status = headset.status;

    if (headset.cfu_context != null && !headset.cfu_context.isNull()) {
        logger.info('handle_unplugged_headset 2a');
        const isConnected = is_device_connected(headset);
        logger.info('handle_unplugged_headset 2b', isConnected);
        try {
            libcfu.cfu_close(headset.cfu_context);
        } catch (e) {
            logger.error('cfu_close error');
            logger.error(e);
        }
        logger.info('handle_unplugged_headset 2c');
    }

    logger.info('handle_unplugged_headset 3');

    headset.is_qcc = false;
    headset.cfu_context = null;
    headset.status = HEADSET_STATUS.HEADSET_DISCONNECTED;
    headset.dev_serial = null;
    headset.dev_id = null;
    headset.flashing_fw = false;

    /* if we're still getting connected, hold on reporting headset is unplugged */
    if (
        old_status == HEADSET_STATUS.HEADSET_GETTING_CONNECTED &&
        headset.time_since_opened <= DEVICE_CONNECTING_MAX_INTERVALS
    ) {
        logger.info(
            'handle_unplugged_headset: time_since_opened %d, need to wait before reporting unplugged',
            headset.time_since_opened
        );
        /* remember we need to report unplugged later */
        headset.report_unplugged = true;
        /* just report progress for now */
        headset.connectionCallback(
            HEADSET_STATUS.HEADSET_GETTING_CONNECTED,
            false /* is DFU */,
            false /* is OLD */
        );
    } else {
        logger.info('handle_unplugged_headset: report unplugged');
        headset.progress = 0;
        headset.progress_scalar = 100 / DEVICE_CONNECTING_MAX_INTERVALS;
        headset.connectionCallback(
            HEADSET_STATUS.HEADSET_DISCONNECTED,
            false /* is DFU */,
            false /* is OLD */
        );
    }

    logger.info('handle_unplugged_headset 3');

    /* and start checking if headset was re-plugged */
    retry_cfu_open(headset);
}

function handle_dfu_headset(headset) {
    let old_status = headset.status;
    headset.status = HEADSET_STATUS.HEADSET_REAL_DFU;

    logger.info('handle_dfu_headset');

    if (old_status != HEADSET_STATUS.HEADSET_REAL_DFU) {
        logger.info('handle_dfu_headset: let app know');
        headset.connectionCallback(
            HEADSET_STATUS.HEADSET_CONNECTED,
            true /* dfu */,
            false /* unsupported */
        );
    }

    retry_cfu_status(headset);
}

function fix_pskey42(headset) {
    //erronous pskey 42 value is "0049 0049 0042 0048 005a 0055 0048 0046 000b 0010 0004 000b 0014 001e 0018 0005"
    var bad_value =
        '0049004900420048005a005500480046000b00100004000b0014001e00180005';
    var buf = Buffer.alloc(PSKEY_SIZE.PSKEY_TO_RESET * 2);
    var ret = libcfu.cfu_read_pskey(
        headset.cfu_context,
        PSKEY_TO_RESET,
        buf,
        PSKEY_SIZE.PSKEY_TO_RESET
    );
    if (ret < 0) {
        logger.info(
            'cfu_read_pskey: error reading pskey42, it might be deleted: %d',
            ret
        );
        return;
    }

    buf.swap16();
    var str_value = buf.toString('hex');
    logger.info('PSKey42: %s', str_value);
    logger.info('PSKey42 size: %d words', ret);
    if (ret != PSKEY_SIZE.PSKEY_TO_RESET) {
        logger.info(
            'fix_pskey42: key is too short (expected %d words)',
            PSKEY_SIZE.PSKEY_DEV_UID
        );
        return;
    }

    if (str_value != bad_value) {
        logger.info("fix_pskey42: key value isn't erroneous");
        return;
    }

    if (
        headset.dev_id != 0x4645 /* FREECOM2PLUS */ &&
        headset.dev_id != 0x4650 /* FREECOM1PLUS */ &&
        headset.dev_id != 0x4634 /* FREECOM4 */
    ) {
        logger.info("device isn't FC1+ or FC2+ or FC4 so not clearing pskey42");
        return;
    }

    logger.info('pskey42 is erroneous, clearing');

    ret = libcfu.cfu_clear_pskey(headset.cfu_context, PSKEY_TO_RESET);
    if (ret != 0) {
        logger.info('cfu_clear_pskey error: %d', ret);
        return;
    }

    return;
}

function handle_connected_headset_serial(headset, buf, err, ret) {
    logger.info('handle_connected_headset_serial: ret', ret, 'err', err);
    if (err) {
        logger.info('error reading serial number');
    } else if (ret !== PSKEY_SIZE.PSKEY_DEV_UID) {
        logger.info(
            'handle_connected_headset: Expected %d words, but received: %d',
            PSKEY_SIZE.PSKEY_DEV_UID,
            ret
        );
    } else {
        parse_pskey_dev_uid(headset, buf);
        const key = headset.dev_serial + '-healthy';
        let healthiness = settings.hasSync(key) ? settings.getSync(key) : true;
        logger.info(
            'handle_connected_headset: device ',
            headset.dev_serial,
            'connected. healthiness:',
            healthiness
        );
        if (!healthiness && !headset.is_qcc) {
            logger.info('switching to DFU mode');
            headset.status = HEADSET_STATUS.HEADSET_SIMULATED_DFU;
        }
    }

    if (!headset.is_qcc) {
        fix_pskey42(headset);
    }

    let is_dfu =
        headset.status == HEADSET_STATUS.HEADSET_REAL_DFU ||
        headset.status == HEADSET_STATUS.HEADSET_SIMULATED_DFU;

    logger.info(
        'handle_connected_headset: update app. status %d, is_dfu: %d',
        headset.status,
        is_dfu
    );

    headset.connectionCallback(
        HEADSET_STATUS.HEADSET_CONNECTED,
        is_dfu,
        false /* unsupported */
    );
}

function handle_connected_headset(headset) {
    let old_connection_status = headset.status;

    logger.info('handle_connected_headset; is QCC = ', headset.is_qcc);

    headset.status = HEADSET_STATUS.HEADSET_CONNECTED;

    //If we weren't connected beforehand, read serial number and let app know we are connected now
    if (old_connection_status != HEADSET_STATUS.HEADSET_CONNECTED) {
        var buf = Buffer.alloc(2 * PSKEY_SIZE.PSKEY_DEV_UID);

        if (headset.is_qcc) {
            logger.info('handle_connected_headset; switching_mode = true');
            headset.switching_mode = true;
            libcfu.cfu_get_config.async(
                headset.cfu_context,
                CONF_BLOCK_DEV_UID,
                buf,
                PSKEY_SIZE.PSKEY_DEV_UID,
                function(err, ret) {
                    logger.info(
                        'handle_connected_headset; switching_mode = false'
                    );
                    headset.switching_mode = false;
                    handle_connected_headset_serial(headset, buf, err, ret);
                }
            );
        } else {
            libcfu.cfu_read_pskey.async(
                headset.cfu_context,
                PSKEY_DEV_UID,
                buf,
                PSKEY_SIZE.PSKEY_DEV_UID,
                function(err, ret) {
                    handle_connected_headset_serial(headset, buf, err, ret);
                }
            );
        }
    }

    retry_cfu_status(headset);
}

function cfu_status_cb(headset, err, status) {
    if (err) {
        logger.info(
            'cfu_status_cb: err %d status %d time_since_opened %d',
            err,
            status,
            headset.time_since_opened
        );
        throw err;
    }

    // During FW flashing, don't invoke callbacks or change state
    if (headset.flashing_fw) {
        logger.info('cfu_status_cb: flashing, so ignoring status for now');
        retry_cfu_status(headset);
        return;
    }

    /* everytime we get a status back, it means another USB device polling interval has passed (except the first) */
    if (headset.time_since_opened <= DEVICE_CONNECTING_MAX_INTERVALS) {
        headset.time_since_opened += 1;
        headset.progress += headset.progress_scalar;
    }

    logger.info(
        'cfu_status_cb: %d = %s status %d time_since_opened %d progress %d progress_scalar %d',
        status,
        DEV_MODE[status],
        headset.status,
        headset.time_since_opened,
        headset.progress,
        headset.progress_scalar
    );

    //cfu status 0 is unplugged
    if (status == 0) {
        headset.cachedHardwareModel = null;
        handle_unplugged_headset(headset);
        return;
    }

    if (headset.time_since_opened <= DEVICE_CONNECTING_MIN_INTERVALS) {
        logger.info('headset is getting connected');
        headset.status = HEADSET_STATUS.HEADSET_GETTING_CONNECTED;
        /* let app know we are getting connected and report progress */
        headset.connectionCallback(
            HEADSET_STATUS.HEADSET_GETTING_CONNECTED,
            false,
            false
        );
        retry_cfu_status(headset);
        return;
    }

    //cfu status 2 is DFU
    if (status == 2) {
        handle_dfu_headset(headset);
        return;
    }

    // QCC device
    headset.is_qcc = is_qcc_device();

    //otherwise, we have a connected headset
    handle_connected_headset(headset);
}

function cfu_open_cb(headset, err, cfu) {
    if (err) {
        logger.error('cfu_open_cb:', err);
        throw err;
    } else if (headset === null) {
        logger.error('cfu_open_cb: headsdet is null');
        return;
    }
    logger.info('cfu_open_cb ', cfu);

    // If no valid device is opened, see if there are any unsupported devices present
    if (cfu.isNull()) {
        logger.info(
            'cfu_open: no device. status %d time_since_opened %d progress %d report_unplugged %d',
            headset.status,
            headset.time_since_opened,
            headset.progress,
            headset.report_unplugged
        );

        const isConnected = is_device_connected(headset);
        logger.info('cfu_open: is_device_connected = ', isConnected);

        if (headset.report_unplugged) {
            /* everytime we get a status back, it means another USB device polling interval has passed (except the first) */
            if (headset.time_since_opened < DEVICE_CONNECTING_MAX_INTERVALS) {
                headset.time_since_opened += 1;
                headset.progress += headset.progress_scalar;
                logger.info(
                    'pretend device is still getting connected and report progress'
                );
                headset.connectionCallback(
                    HEADSET_STATUS.HEADSET_GETTING_CONNECTED,
                    false /* is DFU */,
                    false /* is OLD */
                );
            } else {
                logger.info('tell app that device is unplugged');
                headset.connectionCallback(
                    HEADSET_STATUS.HEADSET_DISCONNECTED,
                    false /* is DFU */,
                    false /* is OLD */
                );
                headset.report_unplugged = false; /* report only once */
                headset.switching_mode = false;
                headset.progress = 0;
                headset.progress_scalar = 100 / DEVICE_CONNECTING_MAX_INTERVALS;
            }
        } else {
            maybe_report_old_headset(headset);
        }

        retry_cfu_open(headset);
        return;
    }

    headset.cfu_context = cfu;
    headset.time_since_opened = 0;
    headset.report_unplugged = false;
    headset.progress_scalar =
        (100 - headset.progress) / DEVICE_CONNECTING_MAX_INTERVALS;

    logger.info(
        'cfu_open: device detected, status %d progress %d progress_scalar %d',
        headset.status,
        headset.progress,
        headset.progress_scalar
    );

    libcfu.cfu_status.async(headset.cfu_context, function(err, res) {
        cfu_status_cb(headset, err, res);
    });
}

class Headset {
    constructor(userConnectionCallback) {
        logger.info(util.format('headset constructor: %s', current_app));
        this.status = HEADSET_STATUS.HEADSET_DISCONNECTED;
        this.flashing_fw = false;
        this.timer = null;
        this._currentPskeyConfiguration = null;
        this._userConnectionCallback = userConnectionCallback;
        this._oldDeviceConnected = false;
        this.dev_serial = null;
        let headsetInstance = this;
        this.time_since_opened = 0;
        this.report_unplugged = false;
        this.progress = 0;
        this.progress_scalar = 100 / DEVICE_CONNECTING_MAX_INTERVALS;
        this.cachedHardwareModel = null;
        this.running_cfu_reset = false;
        this.switching_mode = false;

        retry_cfu_open(headsetInstance);
    }

    connectionCallback(isConnected, isDfu, isOld) {
        this.progress = this.progress > 100 ? 100 : this.progress;
        logger.info(
            'invoking callback: connected %d, dfu %d, old %d, progress %d (time_since_opened %d)',
            isConnected,
            isDfu,
            isOld,
            this.progress,
            this.time_since_opened
        );
        this._userConnectionCallback(isConnected, isDfu, isOld, this.progress);
    }

    destroy() {
        logger.info('headset.destroy()');
        if (this.timer) {
            clearTimeout(this.timer);
            this.timer = null;
        }

        if (this.cfu_context != null && !this.cfu_context.isNull()) {
            const isConnected = is_device_connected(this);
            logger.info('headset.destroy - cfu_close ', isConnected);
            try {
                libcfu.cfu_close(this.cfu_context);
            } catch (e) {
                logger.error('cfu_close error');
                logger.error(e);
            }
            logger.info('headset.destroy - cfu_close 2');
            this.cfu_context = null;
            this.is_qcc = false;
            this.switching_mode = false;
        }
    }

    _getDeviceUniqueID() {
        logger.info('_getDeviceUniqueID');
        var buf = Buffer.alloc(2 * PSKEY_SIZE.PSKEY_DEV_UID);
        logger.info('_getDeviceUniqueID 2', this.is_qcc);

        var ret;
        if (this.is_qcc) {
            ret = libcfu.cfu_get_config(
                this.cfu_context,
                CONF_BLOCK_DEV_UID,
                buf,
                PSKEY_SIZE.PSKEY_DEV_UID
            );
        } else {
            ret = libcfu.cfu_read_pskey(
                this.cfu_context,
                PSKEY_DEV_UID,
                buf,
                PSKEY_SIZE.PSKEY_DEV_UID
            );
        }
        logger.info('_getDeviceUniqueID 3');
        if (ret != PSKEY_SIZE.PSKEY_DEV_UID) {
            console.log(
                'getHardwareModel: Expected %d words, but received: %d',
                PSKEY_SIZE.PSKEY_DEV_UID,
                ret
            );
            return [MODEL_DEVICE_ERROR, 'ERROR'];
        }

        return parse_pskey_dev_uid(this, buf);
    }

    _getBTName() {
        logger.info('_getBTName 1');
        var buf = Buffer.alloc(
            2 *
                (this.is_qcc
                    ? PSKEY_SIZE.QCC_BT_NAME
                    : PSKEY_SIZE.PSKEY_BT_NAME)
        );
        var ret;

        logger.info('_getBTName 2');
        if (this.is_qcc) {
            logger.info('_getBTName 2b');
            ret = libcfu.cfu_get_config(
                this.cfu_context,
                CONF_BLOCK_BT_NAME,
                buf,
                PSKEY_SIZE.QCC_BT_NAME
            );
        } else {
            ret = libcfu.cfu_read_pskey(
                this.cfu_context,
                PSKEY_BT_NAME,
                buf,
                PSKEY_SIZE.PSKEY_BT_NAME
            );
        }

        if (ret < 0) {
            return null;
        }

        var btName = '';
        for (var i = 0; i < ret * (this.is_qcc ? 2 : 2); i += 2) {
            var char = '';
            if (this.is_qcc) {
                var char1Int = buf.readInt8(i + 1),
                    char2Int = buf.readInt8(i);
                if (char1Int !== 0) {
                    char = String.fromCharCode(char1Int);
                }
                if (char2Int !== 0) {
                    char += String.fromCharCode(char2Int);
                }
            } else {
                var charInt = buf.readInt16LE(i);
                char = String.fromCharCode(charInt);
            }
            logger.info('AAAA: ' + i + ': ' + char);
            btName += char;
        }
        logger.info('_getBTName: ' + btName + ' (' + ret + ' words)');
        return btName;
    }

    _getDeviceLanguage() {
        var buf;
        var ret;

        buf = Buffer.alloc(2 * PSKEY_SIZE.QCC_RESERVED);
        ret = libcfu.cfu_get_config(
            this.cfu_context,
            CONF_BLOCK_RESERVED,
            buf,
            PSKEY_SIZE.QCC_RESERVED
        );

        if (ret < 0) {
            return null;
        }

        var language = buf.readInt16LE(0);
        logger.info(
            `_getDeviceLanguage: ${language} => ${QCC_LANGUAGE[language]}`
        );
        return QCC_LANGUAGE[language];
    }

    _getBTMVersion() {
        var buf;
        var ret;

        if (this.is_qcc) {
            buf = Buffer.alloc(2 * PSKEY_SIZE.QCC_PUB_FW_VER);
            ret = libcfu.cfu_get_config(
                this.cfu_context,
                CONF_BLOCK_VERSIONS,
                buf,
                PSKEY_SIZE.QCC_PUB_FW_VER
            );
        } else {
            buf = Buffer.alloc(2 * PSKEY_SIZE.PSKEY_BTM_VERSION);
            ret = libcfu.cfu_read_pskey(
                this.cfu_context,
                PSKEY_BTM_VERSION,
                buf,
                PSKEY_SIZE.PSKEY_BTM_VERSION
            );
        }

        if (ret < 0) {
            return null;
        }

        var btm_fw_ver = buf.readInt16LE(2);
        var btm_fw_subver = buf.readInt16LE(4);
        var btm_fw_rev = buf.readInt16LE(6);
        const btName = `${btm_fw_ver}.${btm_fw_subver}.${btm_fw_rev}`;

        logger.info(`_getBTMVersion: ${btName}`);
        return btName;
    }

    _getPublicFwVersion() {
        var buf;
        var ret;

        if (this.is_qcc) {
            buf = Buffer.alloc(2 * PSKEY_SIZE.QCC_PUB_FW_VER);
            ret = libcfu.cfu_get_config(
                this.cfu_context,
                CONF_BLOCK_VERSIONS,
                buf,
                PSKEY_SIZE.QCC_PUB_FW_VER
            );
        } else {
            buf = Buffer.alloc(2 * PSKEY_SIZE.PSKEY_PUB_FW_VER);
            ret = libcfu.cfu_read_pskey(
                this.cfu_context,
                PSKEY_PUB_FW_VER,
                buf,
                PSKEY_SIZE.PSKEY_PUB_FW_VER
            );
        }

        if (!this.is_qcc && ret != PSKEY_SIZE.PSKEY_PUB_FW_VER) {
            // For older FW versions of Packtalk and Smartpack, version key is smaller - retry reading with a smaller buffer
            buf = Buffer.alloc(PSKEY_SIZE.PSKEY_PUB_FW_VER);
            ret = libcfu.cfu_read_pskey(
                this.cfu_context,
                PSKEY_PUB_FW_VER,
                buf,
                PSKEY_SIZE.PSKEY_PUB_FW_VER / 2
            );
            if (ret != PSKEY_SIZE.PSKEY_PUB_FW_VER / 2) {
                return '????.????';
            }
        }

        var dev_fw_ver = this.is_qcc ? buf.readInt16LE(8) : buf.readInt16LE(0);
        var dev_fw_subver = this.is_qcc
            ? buf.readInt16LE(10)
            : buf.readInt16LE(2);
        var dev_fw_ver_full = dev_fw_ver + '.' + dev_fw_subver;

        return dev_fw_ver_full;
    }

    // Returns the type of hardware, its serial number and its current fw version
    // Preliminary implementation; should be async, use worker pools, and cache results
    getHardwareModel() {
        logger.info('getHardwareModel');

        if (this.cfu_context == null) {
            logger.info('getHardwareModel: Device is unplugged');
            return {
                model: 'No Device',
                serial: 'No device',
                fw_version: 'No device',
                bt_name: 'No device'
            };
        } else if (this.isDfu()) {
            logger.info(
                'getHardwareModel: Device is in DFU mode - cannot read model details'
            );
            return {
                wasInterrupted: true,
                model: MODEL_DEVICE_UNKNOWN,
                serial: MODEL_DEVICE_UNKNOWN,
                fw_version: MODEL_DEVICE_UNKNOWN,
                bt_device: MODEL_DEVICE_UNKNOWN
            };
        } else if (
            (this.cachedHardwareModel &&
                this.cachedHardwareModel !== null &&
                this.isConnected()) ||
            (this.running_cfu_reset &&
                this.cachedHardwareModel &&
                this.cachedHardwareModel !== null)
        ) {
            logger.info(
                'Returning cached hardware model: ' + this.cachedHardwareModel
            );

            var headset = this;
            setTimeout(() => {
                if (headset.running_cfu_reset) return;
                if (
                    headset.cfu_context === null ||
                    headset.cfu_context.isNull()
                )
                    return;

                var status = libcfu.cfu_status(headset.cfu_context);
                logger.info(
                    'getHardwareModel, Status (cached mode) ',
                    status,
                    DEV_MODE[status],
                    headset.running_cfu_reset
                );

                // Switch back to QBTMNormal mode
                if (status === 6 && !this.running_cfu_reset) {
                    setTimeout(() => {
                        if (
                            headset.cfu_context === null ||
                            headset.cfu_context.isNull() ||
                            headset.running_cfu_reset
                        )
                            return;

                        // Do this in the background, since it takes a few seconds and might block the app
                        headset.running_cfu_reset = true;
                        headset.switching_mode = true;
                        logger.info(
                            'getHardwareModel - 1; switching_mode = true'
                        );
                        logger.info(
                            'Calling cfu_reset from getHardwareModel (cached mode)',
                            headset.cfu_context
                        );

                        var ret = libcfu.cfu_reset.async(
                            headset.cfu_context,
                            function(err, status) {
                                logger.info(
                                    'Inside calling cfu_reset from getHardwareModel (cached mode)',
                                    err
                                );
                                headset.running_cfu_reset = false;
                                logger.info(
                                    'getHardwareModel - 1; switching_mode = false'
                                );
                                headset.switching_mode = false;
                            }
                        );
                    }, 100);
                }
            }, 100);

            return this.cachedHardwareModel;
        }

        logger.info('getHardwareModel, Before calling APIs');

        var status = libcfu.cfu_status(this.cfu_context);
        logger.info('getHardwareModel, Status ', status, DEV_MODE[status]);

        var [dev_name, dev_serial] = this._getDeviceUniqueID();
        logger.info(
            'getHardwareModel, After _getDeviceUniqueID',
            dev_name,
            dev_serial
        );

        status = libcfu.cfu_status(this.cfu_context);
        logger.info('getHardwareModel, Status 2 ', status, DEV_MODE[status]);

        if (this.is_qcc) {
            if (status !== 6) {
                logger.error('getHardwareModel, Not in QBTMConfig mode (2) ');
                return {
                    model: 'No Device',
                    serial: 'No device',
                    fw_version: 'No device',
                    bt_name: 'No device',
                    fw_error: true
                };
            }
        }

        var dev_fw_ver_full = this._getPublicFwVersion();
        logger.info(
            'getHardwareModel, After _getPublicFwVersion',
            dev_fw_ver_full
        );
        var bt_name = this._getBTName();
        logger.info('getHardwareModel, After _getBTName');
        const btm_version = this._getBTMVersion();
        logger.info('getHardwareModel, After _getBTMVersion');
        let device_language = null;
        if (this.is_qcc) {
            device_language = this._getDeviceLanguage();
            this.device_language = device_language;
            logger.info('getHardwareModel, After _getDeviceLanguage');
        }

        logger.info('getHardwareModel, After calling APIs ');

        status = libcfu.cfu_status(this.cfu_context);
        logger.info(
            'getHardwareModel, Status after ',
            status,
            DEV_MODE[status]
        );

        if (this.is_qcc && status === 6) {
            // QBTMConfig mode - need to switch to QBTMNormal model
            var context = this.cfu_context;
            var headset = this;
            setTimeout(() => {
                if (
                    headset.cfu_context === null ||
                    headset.cfu_context.isNull() ||
                    headset.running_cfu_reset
                )
                    return;

                // Do this in the background, since it takes a few seconds and might block the app
                headset.running_cfu_reset = true;
                logger.info('getHardwareModel - 2; switching_mode = true');
                headset.switching_mode = true;
                logger.info('Calling cfu_reset from getHardwareModel');

                var ret = libcfu.cfu_reset.async(headset.cfu_context, function(
                    err,
                    status
                ) {
                    logger.info(
                        'Inside calling cfu_reset from getHardwareModel'
                    );
                    status = libcfu.cfu_status(context);
                    headset.running_cfu_reset = false;
                    logger.info('getHardwareModel - 2; switching_mode = false');
                    headset.switching_mode = false;

                    logger.info(
                        'getHardwareModel, Status after switching ',
                        err,
                        status,
                        DEV_MODE[status]
                    );
                });
            }, 100);
        }

        var result = {
            model: dev_name,
            device_language: device_language,
            serial: dev_serial,
            fw_version: dev_fw_ver_full,
            bt_name: bt_name,
            btm_version,
            not_supported:
                dev_name === MODEL_DEVICE_NOT_SUPPORTED ||
                this._oldDeviceConnected,
            error: dev_name === MODEL_DEVICE_ERROR
        };

        this.cachedHardwareModel = result;
        logger.info('getHardwareModel: ', result);

        return result;
    }

    // Returns true if a device is connected, and false otherwise
    isConnected() {
        logger.info('status: %d', this.status);
        return this.status == HEADSET_STATUS.HEADSET_CONNECTED;
    }

    isDfu() {
        return this.status == HEADSET_STATUS.HEADSET_REAL_DFU;
    }

    _extractFirmware(fw, folder) {
        return new Promise((resolve, reject) => {
            tarball.extractTarball(fw, folder, err => {
                if (err) {
                    logger.info(
                        util.format("Can't extract tarball %s: %s", fw, err)
                    );
                    reject(err);
                } else {
                    resolve();
                }
            });
        });
    }

    _flashImage(
        progress_base,
        progress_goal,
        progress_callback,
        flasher,
        cfu,
        data,
        len
    ) {
        return new Promise((resolve, reject) => {
            var cfu_progress_cb = ffi.Callback(
                'void',
                ['uint64', 'uint64'],
                function(cur, total) {
                    let progress_increase = (progress_goal * cur) / total;
                    let total_progress = Math.round(
                        progress_base + progress_increase
                    );
                    progress_callback(total_progress);
                }
            );
            var cfu_complete_cb = function(err, ret) {
                if (err) {
                    logger.info(util.format('cfu_flash_cb error: %s', err));
                    reject(err);
                }
                if (ret) {
                    logger.info(util.format('cfu_flash_cb failed: %d', ret));
                    reject(Error(ret));
                }
                resolve(ret);
                // Make an extra reference to the callback pointer to avoid GC, see ffi docs
                cfu_progress_cb;
            };

            flasher(cfu, data, len, cfu_progress_cb, cfu_complete_cb);
        });
    }

    // Loads the firmware specified in the path onto the device. Periodically calls progress_callback() with a value between 0-100
    // to indicate the progress of the update. This function returns a promise which can be awaited upon. Once resolved, it returns
    // a return code indicating the status of the update. 0 indicates success, any other value indicates an error.
    async loadFW(fw_path, progress_callback) {
        this.flashing_fw = true;
        logger.info('loadFW: ' + fw_path);

        //We always flash at least the app.dfu image
        var images_to_flash = [this.is_qcc ? 'ver.bin' : 'app.dfu'];

        //Extract tarball to tmp folder, remove it on exit even if not empty
        var tmpfolder = tmp.dirSync({ unsafeCleanup: true });
        await this._extractFirmware(fw_path, tmpfolder.name);

        //do we also have a dsp image to flash?
        var dsp_img = path.join(tmpfolder.name, 'dsp.img');
        if (fs.existsSync(dsp_img)) {
            logger.info(util.format('found %s', dsp_img));
            images_to_flash.push('dsp.img');
        } else {
            logger.info('no dsp.img this time');
        }

        //do we also have a dsp loader to flash?
        var loader_name = path.join(tmpfolder.name, 'ldr.bin');
        if (fs.existsSync(loader_name)) {
            logger.info(util.format('found %s', loader_name));
            images_to_flash.push('ldr.bin');
        } else {
            logger.info('no ldr.bin this time');
        }

        //do we also have an external flash to update?
        var external_pack = path.join(tmpfolder.name, 'ext.pck');
        if (fs.existsSync(external_pack)) {
            logger.info(util.format('found %s', external_pack));
            images_to_flash.push('ext.pck');
        } else {
            logger.info('no ext.pck this time');
        }

        logger.info(
            util.format('Going to flash %d images', images_to_flash.length)
        );
        var progress_per_image = 100 / images_to_flash.length;
        var progress_base = 0;

        //we're beginning flashing so mark this device firmware state as non-healthy
        if (!this.dev_serial) {
            logger.info(
                'loadFW: null serial number, not marking fw health status'
            );
        } else {
            logger.info('loadFW: marking', this.dev_serial, 'as non-healthy');
            settings.setSync(this.dev_serial + '-healthy', false);
        }

        if (this.is_qcc) {
            const isConnected = is_device_connected(this);
            if (!isConnected) {
                logger.info('loadFW: Device disconnected');
                throw Error('Device disconnected while flashing FW');
            }

            var status = libcfu.cfu_status(this.cfu_context);
            if (status === 5) {
                // QBTMNormal mode - need to switch to QBTMConfig mode before flashing
                logger.info('loadFW: Switching to QBTMConfig before flashing');
                this.switching_mode = true;
                this.running_cfu_reset = true;
                var ret = libcfu.cfu_reset(this.cfu_context);
                this.switching_mode = false;
                this.running_cfu_reset = false;
                status = libcfu.cfu_status(this.cfu_context);
                logger.info('loadFW: Status:', status, DEV_MODE[status]);
            }
        }

        for (var image of images_to_flash) {
            logger.info(util.format('Flashing image %s, please wait', image));
            var fullname = path.join(tmpfolder.name, image);

            try {
                var data = await fsPromises.readFile(fullname);
            } catch (err) {
                logger.info("can't read image: " + err);
                this.flashing_fw = false;
                throw err;
            }

            try {
                let ret = await this._flashImage(
                    progress_base,
                    progress_per_image,
                    progress_callback,
                    FLASHER_FUNC[image],
                    this.cfu_context,
                    data,
                    data.length
                );
                logger.info(
                    util.format(
                        'Finished flashing image %s, ret: %d',
                        image,
                        ret
                    )
                );
            } catch (e) {
                logger.info('Error flashing image - ' + e);
                this.flashing_fw = false;
                throw e;
            }

            progress_base += progress_per_image;
        }

        logger.info(
            util.format('Finished flashing %d images', images_to_flash.length)
        );
        this.flashing_fw = false;

        //we've finished flashing successfully so mark this device firmware state as healthy again
        this._getDeviceUniqueID();
        logger.info('loadFW: marking', this.dev_serial, 'as healthy');
        settings.setSync(this.dev_serial + '-healthy', true);
        this.status = HEADSET_STATUS.HEADSET_CONNECTED;
        this.cachedHardwareModel = null;

        // Tell the UI the device is connected
        this.connectionCallback(
            HEADSET_STATUS.HEADSET_CONNECTED,
            false /* is DFU */,
            false /* is OLD */
        );

        var status = libcfu.cfu_status(this.cfu_context);
        if (this.is_qcc && status === 6) {
            // QBTMConfig mode - need to switch to QBTMNormal mode after flashing
            logger.info('Switching to QBTMNormal after flashing');
            var context = this.cfu_context;
            this.running_cfu_reset = true;
            var headset = this;
            this.switching_mode = true;
            libcfu.cfu_reset.async(context, function(err, status) {
                headset.switching_mode = false;
                status = libcfu.cfu_status(context);
                headset.running_cfu_reset = false;

                logger.info(
                    'Status after switching ',
                    err,
                    status,
                    DEV_MODE[status]
                );
            });
        }

        return 0;
    }
}

module.exports = Headset;
